c     -----------------------------------------------------
c     Copyright (c) Harry Kooijman, April 1993.
c     -----------------------------------------------------
c     Spatial differentiation routines for Method Of Lines.
c     This implementation utilizes two simple routines to
c     compute the finite difference approximations.  This
c     sacrifies some speed for readability and ease of
c     coding.  If needed, Jacobian information can be
c     generated as well to speed up the calculations.
c
c     f1234: 1 dimension, 2nd derivative, 3rd order,
c            4 points spatial differentiation
c     xl,xu: Lower/upper integration bounds
c     u:     Variable
c     ux:    Approximation of 1st derivative
c     uxx:   Approximation of 2nd derivative
c     dux:   Jacobian of 1st derivative approximation
c     duxx:  Jacobian of 2nd derivative approximation
c     nl,nu: Lower/upper boundary conditions;
c            Dirichlet=1, Neumann=2 (ux1,uxn)
c     dod:   Logical to flag 'DO Derivative'
c     n:     Number of lines
c
c     -----------------------------------------------------



      double precision function sdux(term,coef,u,inum)
      integer          inum
      double precision term,coef(inum),u(inum)
c
      integer          i
      double precision ux
c
      ux=coef(1)*u(1)
      do i=2,inum
        ux=ux+coef(i)*u(i)
      enddo
      sdux=term*ux
c
      return
      end



      subroutine sddux(term,coef,dux,istart,inum,i,n)
      integer          istart,inum,i,n
      double precision term,coef(inum),dux(n,n)
c
      integer          j,k
c
      j=istart
      do k=1,inum
        dux(i,j)=term*coef(k)
        j=j+1
      enddo
c
      return
      end



      subroutine f1123(xl,xu,n,u,ux,dux,dod)
c
c     Three point, 2nd order finite difference
c     approximation for first derivative in one dimension
c
      integer          n
      double precision xl,xu,u(n),ux(n),dux(n,n)
      logical          dod
c
      integer          nm1,i
      double precision term,sdux
      double precision c1(3),c2(3),c3(3)
      data  c1         /-3.d0,+4.d0,-1.d0/
      data  c2         /-1.d0, 0.d0,+1.d0/
      data  c3         /+1.d0,-4.d0,+3.d0/
c
      nm1=n-1
      term=dble(nm1)/((xu-xl)*2.d0)
c
      ux(1)=sdux(term,c1,u(1),3)
      do i=2,nm1
        ux(i)=sdux(term,c2,u(i-1),3)
      enddo
      ux(n)=sdux(term,c3,u(n-2),3)
c
      if (dod) then
        call mat0(dux,n,n)
        call sddux(term,c1,dux,1,3,1,n)
        do i=2,nm1
          call sddux(term,c2,dux,i-1,3,i,n)
        enddo
        call sddux(term,c3,dux,n-2,3,n,n)
      endif
c
      return
      end



      subroutine f1145(xl,xu,n,u,ux,dux,dod)
c
c     Five point, 4th order finite difference
c     approximation for first derivative in one dimension
c
      integer          n
      double precision xl,xu,u(n),ux(n),dux(n,n)
      logical          dod
c
      integer          nm1,nm2,i
      double precision term,sdux
      double precision c1(5),c2(5),c3(5),c4(5),c5(5)
      data  c1         /-50.d0,+96.d0,-72.d0,+32.d0, -6.d0/
      data  c2         / -6.d0,-20.d0,+36.d0,-12.d0, +2.d0/
      data  c3         / +2.d0,-16.d0,  0.d0,+16.d0, -2.d0/
      data  c4         / -2.d0,+12.d0,-36.d0,+20.d0, +6.d0/
      data  c5         / +6.d0,-32.d0,+72.d0,-96.d0,+50.d0/
c
      nm1=n-1
      nm2=n-2
      term=dble(nm1)/((xu-xl)*24.d0)
c
      ux(1)=sdux(term,c1,u(1),5)
      ux(2)=sdux(term,c2,u(1),5)
      do i=3,nm2
        ux(i)=sdux(term,c3,u(i-2),5)
      enddo
      ux(nm1)=sdux(term,c4,u(n-4),5)
      ux(n)=sdux(term,c5,u(n-4),5)
c
      if (dod) then
        call mat0(dux,n,n)
        call sddux(term,c1,dux,1,5,1,n)
        call sddux(term,c2,dux,1,5,2,n)
        do i=3,nm2
          call sddux(term,c3,dux,i-2,5,i,n)
        enddo
        call sddux(term,c4,dux,n-4,5,nm1,n)
        call sddux(term,c5,dux,n-4,5,n,n)
      endif
c
      return
      end



      subroutine f1226(xl,xu,n,u,ux1,uxn,uxx,duxx,dod,nl,nu)
c
c     Six/Five point, 2nd order finite difference
c     approximation for second derivative in one dimension
c     Boundary conditions: Dirichlet=1, Neumann=2.
c
      integer          n,nl,nu
      double precision xl,xu,u(n),ux1,uxn,uxx(n),duxx(n,n)
      logical          dod
c
      integer          nm1,nm2,i
      double precision dx,term,sdux
      double precision c1a(6),c1b(5),c2(6),c3(5),c4(6),c5a(6),c5b(5),c6
      data c1a         /+45.d0,-154.d0,+214.d0,-156.d0,+61.d0,-10.d0/
      data c1b         /-69.16666666666667d0,+96.d0,-36.d0,
     +                  +10.66666666666667d0,-1.5d0/
      data c2          /10.d0,-15.d0,-4.d0,+14.d0,-6.d0,+1.d0/
      data c3          /-1.d0,+16.d0,-30.d0,+16.d0,-1.d0/
      data c4          /+1.d0,-6.d0,+14.d0,-4.d0,-15.d0,10.d0/
      data c5a         /-10.d0,+61.d0,-156.d0,+214.d0,-154.d0,+45.d0/
      data c5b         /-1.5d0,+10.66666666666667d0,-36.d0,+96.d0,
     +                  -69.16666666666667d0/
      data c6          /-50.d0/
c
      nm1=n-1
      nm2=n-2
      dx=(xu-xl)/dble(nm1)
      term=1.0d0/(12.d0*dx**2)
c
      if (nl .eq. 1) then
c       uxx(1)=sdux(term,c1a,u(1),6)
        uxx(1)=0
      else if (nl .eq. 2) then
        uxx(1)=sdux(term,c1b,u(1),5)+term*c6*dx*ux1
      endif
      uxx(2)=sdux(term,c2,u(1),6)
      do i=3,nm2
        uxx(i)=sdux(term,c3,u(i-2),5)
      enddo
      uxx(nm1)=sdux(term,c4,u(n-5),6)
      if (nu .eq. 1) then
c       uxx(n)=sdux(term,c5a,u(n-5),6)
        uxx(n)=0
      else if (nu .eq. 2) then
        uxx(n)=sdux(term,c5b,u(n-4),5)-term*c6*dx*uxn
      endif
c
      if (dod) then
        call mat0(duxx,n,n)
        if (nl .eq. 1) then
c         call sddux(term,c1a,duxx,1,6,1,n)
        else if (nl .eq. 2) then
          call sddux(term,c1b,duxx,1,5,1,n)
        endif
        call sddux(term,c2,duxx,1,6,1,n)
        do i=3,nm2
          call sddux(term,c3,duxx,i-2,5,i,n)
        enddo
        call sddux(term,c4,duxx,n-5,6,nm1,n)
        if (nu .eq. 1) then
c         call sddux(term,c5a,duxx,n-5,6,n,n)
        else if (nu .eq. 2) then
          call sddux(term,c5b,duxx,n-4,5,n,n)
        endif
      endif
c
      return
      end



      subroutine f1112d(xl,xu,n,u,ux,dux,dod,direct)
c
c     Two point, 1st order directional finite difference
c     approximation for first derivative in one dimension
c
      integer          n
      double precision xl,xu,u(n),ux(n),dux(n,n),direct
      logical          dod
c
      integer          nm1,i
      double precision term,sdux
      double precision c1(2)
      data  c1         /-1.d0,+1.d0/
c
      nm1=n-1
      term=dble(nm1)/(xu-xl)
c
      if (direct .ge. 0.d0) then
        ux(1)=sdux(term,c1,u(1),2)
        do i=2,n
          ux(i)=sdux(term,c1,u(i-1),2)
        enddo
      else
        do i=1,nm1
          ux(i)=sdux(term,c1,u(i),2)
        enddo
        ux(n)=sdux(term,c1,u(nm1),2)
      endif
c
      if (dod) then
        call mat0(dux,n,n)
        if (Direct .ge. 0.d0) then
          call sddux(term,c1,dux,1,2,1,n)
          do i=2,n
            call sddux(term,c1,dux,i-1,2,i,n)
          enddo
        else
          do i=1,nm1
            call sddux(term,c1,dux,i,2,i,n)
          enddo
          call sddux(term,c1,dux,nm1,2,n,n)
        endif
      endif
c
      return
      end



      subroutine f1123d(xl,xu,n,u,ux,dux,dod,direct)
c
c     Three point, 2nd order directional finite difference
c     approximation for first derivative in one dimension
c
      integer          n
      double precision xl,xu,u(n),ux(n),dux(n,n),direct
      logical          dod
c
      integer          nm1,nm2,i
      double precision term,sdux
      double precision c1(3),c2(3),c3(3)
      data  c1         /-3.d0,+4.d0,-1.d0/
      data  c2         /-1.d0,+0.d0,+1.d0/
      data  c3         /+1.d0,-4.d0,+3.d0/
c
      nm1=n-1
      nm2=n-2
      term=dble(nm1)/((xu-xl)*2.d0)
c
      if (direct .ge. 0.0d0) then
        ux(1)=sdux(term,c1,u(1),3)
        ux(2)=sdux(term,c2,u(1),3)
        do i=3,n
          ux(i)=sdux(term,c3,u(i-2),3)
        enddo
      else
        do i=1,nm2
          ux(i)=sdux(term,c1,u(i),3)
        enddo
        ux(nm1)=sdux(term,c1,u(nm2),3)
        ux(n)  =sdux(term,c1,u(nm2),3)
      endif
c
      if (dod) then
        call mat0(dux,n,n)
        if (Direct .ge. 0.d0) then
          call sddux(term,c1,dux,1,3,1,n)
          call sddux(term,c2,dux,1,3,2,n)
          do i=3,n
            call sddux(term,c3,dux,i-2,3,i,n)
          enddo
        else
          do i=1,nm2
            call sddux(term,c1,dux,i,3,i,n)
          enddo
          call sddux(term,c2,dux,nm2,3,nm1,n)
          call sddux(term,c3,dux,nm2,3,n,n)
        endif
      endif
c
      return
      end



      subroutine f1145d(xl,xu,n,u,ux,dux,dod,direct)
c
c     Five point, 4th order directional finite difference
c     approximation for first derivative in one dimension
c
      integer          n
      double precision xl,xu,u(n),ux(n),dux(n,n),direct
      logical          dod
c
      integer          nm1,nm2,nm3,nm4,i
      double precision dx,term,sdux
      double precision c1(5),c2(5),c3(5),c4(5),c5(5)
      data  c1         /-50.d0,+96.d0,-72.d0,+32.d0, -6.d0/
      data  c2         / -6.d0,-20.d0,+36.d0,-12.d0, +2.d0/
      data  c3         / +2.d0,-16.d0, +0.d0,+16.d0, -2.d0/
      data  c4         / -2.d0,+12.d0,-36.d0,+20.d0, +6.d0/
      data  c5         / +6.d0,-32.d0,+72.d0,-96.d0,+50.d0/
c
      nm1=n-1
      nm2=n-2
      nm3=n-3
      nm4=n-4
      dx=(xu-xl)/dble(nm1)
      term=1.d0/(dx*24.d0)
c
      if (direct .ge. 0.0d0) then
        ux(1)=sdux(term,c1,u(1),5)
        ux(2)=sdux(term,c2,u(1),5)
        ux(3)=sdux(term,c3,u(1),5)
        do i=4,nm1
          ux(i)=sdux(term,c4,u(i-3),5)
        enddo
        ux(n)=sdux(term,c5,u(nm4),5)
      else
        ux(1)=sdux(term,c1,u(1),5)
        do i=2,nm3
          ux(i)=sdux(term,c2,u(i-1),5)
        enddo
        ux(nm2)=sdux(term,c3,u(nm4),5)
        ux(nm1)=sdux(term,c4,u(nm4),5)
        ux(n)  =sdux(term,c5,u(nm4),5)
      endif
c
      if (dod) then
        call mat0(dux,n,n)
        if (Direct .ge. 0.d0) then
          call sddux(term,c1,dux,1,5,1,n)
          call sddux(term,c2,dux,1,5,2,n)
          call sddux(term,c3,dux,1,5,3,n)
          do i=4,nm1
            call sddux(term,c4,dux,i-3,5,i,n)
          enddo
          call sddux(term,c5,dux,nm4,5,n,n)
        else
          call sddux(term,c1,dux,1,5,1,n)
          do i=2,nm3
            call sddux(term,c2,dux,i-1,5,i,n)
          enddo
          call sddux(term,c3,dux,nm4,5,nm2,n)
          call sddux(term,c4,dux,nm4,5,nm1,n)
          call sddux(term,c5,dux,nm4,5,n,n)
        endif
      endif
c
      return
      end

