c ------------------------------------------------------------------ c
c                                                                    c
c        BESIRK v1.5: Differential-Algebraic Equation Solver         c
c                     Full matrix version using LU backsubstitution  c
c                     Harry Kooijman, September 1995                 c
c                                                                    c
c ------------------------------------------------------------------ c
c                                                                    c
c Integrate DAE system from 't' to 'tout':                           c
c                                                                    c
c   [B](dY/dt) = F(Y)                                                c
c                                                                    c
c    Y  = Vector of n variables,                                     c
c   [B] = Linear matrix (n*n) of constant coefficients.              c
c         For algebraic equations the rows of [B] are zero.          c
c    F  = Function vector length n                                   c
c                                                                    c
c by using the Michelsen Semi-Implicit Runge-Kutta with a Bulirsch-  c
c Stoer extrapolation method. The "time" invariant system can be     c
c solved as well, using Newton's method:                             c
c                                                                    c
c             (0) = F(Y)                                             c
c                                                                    c
c ------------------------------------------------------------------ c
c                                                                    c
c   call BESIRK(n,info,Y,B,nB,t,tout,treprt,h,hnext,hmin,hmax,       c
c  +            Idid,Nstep,Eval,Update,Toler,Report,NumJac,MaxIt,    c
c  +            Iter,AbsTol,RelTol,AlgEqn,Dw,Iw,kountD,kountI)       c
c                                                                    c
c where: n        = Number of equations/variables                    c
c        info     = Info level flag                                  c
c        Y(n)     = Variables vector                                 c
c        B(nB,nB) = Matrix with coefficients for the differentials   c
c        nB       = Declared size of B                               c
c        t        = Current independent variable, time               c
c        tout     = Wanted value for time                            c
c        treprt   = Next report time                                 c
c        h        = Initial stepsize in time                         c
c        hmin     = Minimum allowed stepsize                         c
c        hmax     = Maximum allowed stepsize                         c
c        Nstep    = Stepsize for numerical Jacobian                  c
c        Eval     = User-supplied problem routine, see below         c
c        Update   = User-supplied update routine, see below          c
c        Toler    = User-supplied convergence check routine, see beloc
c        Report   = User-supplied report routine, see below          c
c        NumJac   = If true than evaluate Jacobian numerically       c
c        MaxIt    = Maximum number of iterations (steps) allowed     c
c        Iter     = Number of iterations (steps) taken               c
c        AbsTol   = Absolute tolerance vector                        c
c        RelTol   = Relative tolerance vector                        c
c        AlgEqn   = If true than control on the error of the         c
c                   algebraic equations, else on error in estimates  c
c                   of the algebraic variables.                      c
c        Dw(*)    = Double precision work space                      c
c        Iw(*)    = Integer work space                               c
c        kountD   = Offset counter for Dw                            c
c        kountI   = Offset counter for Iw                            c
c                                                                    c
c output:Y(n)     = Variables vector                                 c
c        hnext    = Calculated stepsize for next integration         c
c        Idid     = Result of Integration:                           c
c                     -x:User error                                  c
c                      0:Successful integration, no errors           c
c                      1:Did not integrate up to tout (nreach)       c
c                      2:Tolerances are too small (tolerr)           c
c                      3:Nstep is too small (nerr)                   c
c                      4:Failed step                                 c
c                      5:Singular matrix in LU (singul)              c
c                      6:User violation (userv)                      c
c                      7:Next stepsize is too small (nexth)          c
c                     10:Algebraic system                            c
c                     11:Algebraic constraints violated              c
c                   Internal:                                        c
c                      8:Done                                        c
c                      9:Polation error                              c
c                                                                    c
c When h=0 in the call to BESIRK, the stepsize will be computed by   c
c considering the sytem as an ODE and using the simple truncation    c
c error formula of the Euler method. The smallest estimated step is  c
c then corrected for the order of the SIRK and the number of steps   c
c that is desired for the extrapolation. Afterwards the minimum and  c
c maximum limits on the stepsize are applied.                        c
c                                                                    c
c BESIRK options can be set by changing parameters in the BSIRK1     c
c common block. It is declared as:                                   c
c                                                                    c
c     common   /Bsirk1/ Emax,sMax,sUse,sStep,sSeq,                   c
c    +                  PolOrd,PolTyp,cIndex,                        c
c    +                  Shrink,Grow,MaxInc,                          c
c    +                  UpdJac                                       c
c                                                                    c
c with: Emax     = Maximum number in sequence with increasing error  c
c       sMax     = Maximum number in seuence                         c
c       sUse     = Estimate of best number of steps in sequence      c
c       sStep    = Maximum increment of steps for sequence           c
c       sSeq(12) = Sequence of steps                                 c
c       PolOrd   = Extrapolation order in stepsize                   c
c       PolTyp   = Extrapolation type: 0=Polynomial, 1=Rational      c
c       cIndex   = Index check: 0=no check, 1=1st iter, 2=always     c
c       Shrink   = Shrink factor big stepsize                        c
c       Grow     = Grow factor big stepsize                          c
c       MaxInc   = Maximum increment of big stepsize                 c
c       UpdJac   = If true than Jacobian updated every small step    c
c                                                                    c
c BESIRK needs a user supplied subroutine that will evaluate F(Y)    c
c and, if possible, the Jacobian dF(Y)/dY. The routine should look   c
c like:                                                              c
c                                                                    c
c   Subroutine Eval(n,info,F,Y,Jac,t,Jacob,error)                    c
c  +                Dw,Iw,kountD,kountI)                             c
c   integer n,info,error,kountD,kountI,Iw(*)                         c
c   double precision t,F(n),Y(n),Jac(n,n),Dw(*)                      c
c   logical Jacob                                                    c
c                                                                    c
c where:  n        = Number of equations/variables                   c
c         info     = Info level flag                                 c
c         Y(n)     = Variables vector                                c
c         t        = independent variable, time                      c
c         Jacob    = If true Jacobian needs to be calculated         c
c         Dw(*)    = Double precision work space                     c
c         Iw(*)    = Integer work space                              c
c         kountD   = Offset counter for Dw                           c
c         kountI   = Offset counter for Iw                           c
c                                                                    c
c output: F        = Function values vector                          c
c         Jac(n,n) = Jacobian matrix with dF(Y)/dY                   c
c         Error    = Error flag (0 on entry)                         c
c                                                                    c
c If, for any reason, there is a problem evaluating the function     c
c values or Jacobian, than the error flag can be set by the user.    c
c It should be set to a negative value for immediate exit of BESIRK. c
c The only allowed positive number is for signaling a user violation c
c which can be used for implicit discontinuities. BESIRK will try to c
c approach the time at which the violation is set, and then exit     c
c with the user violation as result. Any other positive error value  c
c will cause BESIRK to exit with the *negative* value!               c
c                                                                    c
c The reason for using one routine to evaluate both function values  c
c and Jacobian is that while computing the Jacobian usually the      c
c function values and/or intermediate results can be reused, saving  c
c computation time. It also avoids the use of the wrong jacobian     c
c routine if multiple problems are being solved.                     c
c                                                                    c
c The user can pass arguments to the user routine through the work   c
c space arrays Dw and Iw. The user routine can use the work space    c
c arrays as well for temporary storage. The size of the work space   c
c must be adapted accordingly!                                       c
c                                                                    c
c BESIRK needs three more routines that will allow you to adapt the  c
c algorithm to your needs. One is for updating the variables, one    c
c for checking convergence on the tolerances and one for reporting   c
c intermediate results. The update routine can be useful for inter-  c
c rupting the program or control of the Newton's step. The routine   c
c to check convergence is usually problem dependent. The report      c
c routine can show results, iteration number and errors. It can also c
c set the variable treprt, which is the time where we want BESIRK to c
c integrate to exactly and generate a report. Then in the report     c
c routine a new treprt can be set. Default routines are supplied for c
c general use. The routines are actually implemented as functions:   c
c                                                                    c
c   Integer function Update(n,iter,info,t,h,Y,DeltaY)                c
c                                                                    c
c where:  n         = Number of equations/variables                  c
c         iter      = Iteration counter                              c
c         info      = Info level flag                                c
c         t         = Current independent variable, time             c
c         h         = Stepsize in time                               c
c         Y(n)      = Variables vector                               c
c         DeltaY(n) = Error estimates of variable vector             c
c                                                                    c
c output: Update    = Result of update; 0=OK.                        c
c                     Output's job is to add stepsize h to time t    c
c                     and correction DeltaY to Y.                    c
c                                                                    c
c   Logical function Toler(n,iter,info,t,et,Y,E,A,R)                 c
c                                                                    c
c where:  n         = Number of equations/variables                  c
c         iter      = Iteration counter                              c
c         info      = Info level flag                                c
c         t         = Current independent variable, time             c
c         Y(n)      = Variables vector                               c
c         E(n)      = Error estimates of variable vector             c
c         A(n)      = Absolute tolerances vector                     c
c         R(n)      = Relative tolerances vector                     c
c                                                                    c
c output: et        = (Largest) Error/Tolerance                      c
c         Toler     = True if error within bounds (et<1),            c
c                     false otherwise (et>1)                         c
c                                                                    c
c   Integer function Report(n,iter,info,t,treprt,Y,E)                c
c                                                                    c
c where:  n         = Number of equations/variables                  c
c         iter      = Iteration counter                              c
c         info      = Info level flag                                c
c         t         = Current independent variable, time             c
c         Y(n)      = Variables vector                               c
c         E(n)      = Error estimates of variable vector             c
c                                                                    c
c output: Report    = 0 if all is OK. Use this routine to write the  c
c                     intermeadiate results for later use.           c
c         treprt    = Next report time                               c
c                                                                    c
c The default routines for Update/Toler/Report are: StandU, StandT,  c
c and StandR. They will only print information if the info flag is   c
c larger than zero.  To ease the use of BESIRK a subroutine BSIRK    c
c is included which uses these standard routines, it uses the        c
c following call:                                                    c
c                                                                    c
c   call BSIRK(n,info,Y,B,nB,t,tout,treprt,h,hnext,hmin,hmax,Idid,   c
c  +           Nstep,Eval,NumJac,MaxIt,Iter,AbsTol,RelTol,           c
c  +           Dw,Iw,kountD,kountI)                                  c
c                                                                    c
c To solve the 'time'-invariant problem you can call the Newton      c
c routine that uses the previously discussed Update, Toler and       c
c Report routines as well. It is used in a similar way as BESIRK:    c
c                                                                    c
c   call Newton(n,info,Y,t,Idid,Nstep,Eval,Update,Toler,Report,      c
c  +            NumJac,UpdJac,MaxIt,Iter,AbsTol,RelTol,Fnorm,        c
c  +            Dw,Iw,kountD,kountI)                                 c
c                                                                    c
c where: n      = Number of equations/variables                      c
c        info   = Info level flag (0-10)                             c
c        Y(n)   = Variables vector                                   c
c        t      = Current independent variable, time                 c
c        Nstep  = Stepsize for numerical Jacobian                    c
c        Eval   = User-supplied problem routine                      c
c        Update = User-supplied update routine                       c
c        Toler  = User-supplied convergence check routine            c
c        Report = User-supplied report routine                       c
c        NumJac = If true than evaluate Jacobian numerically         c
c        UpdJac = If true than Jacobian is evaluated at each iterate c
c        MaxIt  = Maximum number of iterations allowed               c
c        Iter   = Number of iterations taken                         c
c        AbsTol = Absolute tolerance vector                          c
c        RelTol = Relative tolerance vector                          c
c        Dw(*)  = Double precision work space                        c
c        Iw(*)  = Integer work space                                 c
c        kountD = Offset counter for Dw                              c
c        kountI = Offset counter for Iw                              c
c                                                                    c
c output:Y(n)     = Variables vector                                 c
c        Fnorm  = 2norm of function vector                           c
c        Idid     = Result of Integration:                           c
c                     -x:User error                                  c
c                      0:Successful integration, no errors           c
c                      1:Reached maximum number of iterations        c
c                      5:Singular matrix in LU (singul)              c
c                                                                    c
c ------------------------------------------------------------------ c
c BESIRK uses work space arrays passed to the routine by the user.   c
c They need to be arrays of doubles (Dw) and integers (Iw) of suf-   c
c ficient size. A good estimate for the minimum size of them is:     c
c                                                                    c
c Routine:  Variables:                          Dw:         Iw:      c
c ---------------------------------------------------------------    c
c BESIRK    F,Yorg,Eps,DidY,Jac       15*n+n*n+smax                  c
c           infoB                                             n      c
c          [cIndex>0]                                      [2*n]     c
c          [AlgEqn=True]                         [n]                 c
c                                                                    c
c Msirk3    k1,k2,k3,FF,Jac                 4*n+n*n                  c
c           inx                                               n      c
c                                                                    c
c LUdcmp    vv                                    n                  c
c   or                                           or                  c
c XXXPol    fx/c                               smax                  c
c                                   --------------- +       --- +    c
c Total:                            20*n+2*n*n+smax         2*n      c
c                                               [+n]       [+2*n]    c
c ---------------------------------------------------------------    c
c                                                                    c
c These totals are the amount BESIRK uses, the use of the work space c
c arrays by the user routines must be added.  On start, the work     c
c space counters kountD and kountI must be set equal to the size     c
c of the work space arrays.  This way it is very easy for BESIRK to  c
c check whether the work space is exhausted, namely if the counter   c
c becomes zero or negative (We start using the workspace at the      c
c end and fill it up to the begining). If there is insufficient      c
c workspace, BESIRK will stop the program and print a message to     c
c prompt you to use larger work space arrays.  The numbers in []     c
c are to be added if the corresponding options are specified.        c
c                                                                    c
c To use the work space arrays you can call the routine Nwork to     c
c "allocate" variable space and Kwork to "deallocate" it.            c
c ------------------------------------------------------------------ c
c BESIRK and associated routines actually use a couple of general    c
c routines that might be handy for you to use as well. Here is a     c
c list of them (they are located at the end of this file):           c
c                                                                    c
c  CpyVec: Copy a vector (size n)                                    c
c  CpyMat: Copy a matrix (size n*n)                                  c
c  Vec0:   Reset a vector (size n)                                   c
c  Mat0:   Reset a matrix (size n*n)                                 c
c  AddVec: Add two vectors (size n)                                  c
c  SubVec: Subtract two vectors (size n)                             c
c  ScVec:  Scale a vector (size n)                                   c
c  ScaMat: Scale a matrix (size n by n)                              c
c  PutMat: Put a vector (size n) in a matrix (size n by n)           c
c                                                                    c
c  From the classic Numerical Recipes in Fortran:                    c
c                                                                    c
c  LUdcmp: LU matrix decomposition                                   c
c  LUbksb: LU back-substitution                                      c
c                                                                    c
c  For the extrapolation to zero stepsize we use two routines that   c
c  are adapted versions of those found in the Numerical Recipes in   c
c  Fortran. They are called:                                         c
c                                                                    c
c  RatPol: Rational extrapolation                                    c
c  PolPol: Polynomial extrapolation                                  c
c                                                                    c
c If you want to solve sparse or banded systems you need to substi-  c
c tute the LU and matrix routines with others, as well as adapt the  c
c allocation of these matrices (for the Jacobians).                  c
c ------------------------------------------------------------------ c
c For those interested in the background of the algorithm read       c
c chapter 16 of the book "Numerical Recipes" (2nd edition).  It does c
c not contain the algorithm in question, but does describe the ideas c
c behind it rather nicely and mentions other references.             c
c ------------------------------------------------------------------ c
c For more information or suggestions send (e-) mail to:             c
c                                                                    c
c kooijman@sun.soe.clarkson.edu                                      c
c                                                                    c
c Harry Kooijman                                                     c
c Department of Chemical Engineering                                 c
c Clarkson University                                                c
c Potsdam, NY 13699-5705                                             c
c ------------------------------------------------------------------ c
c Updates:  7Jul93: - Initial step determination. Order difference   c
c                     was 2/3, set to 1/3. Works better on Pendulum. c
c                   - Added argument AlgEqn to BESIRK to select      c
c                     error control on the algebraic equations or    c
c                     on the algebraic variables (quicker).          c
c                   - Added index indicator routine IIndex.  Will    c
c                     report numbers of algebraic/differential eqns. c
c                     and indication of the index of the system.     c
c                     The check is optional (see cIndex in Bsirk1).  c
c ------------------------------------------------------------------ c



      integer function Nwork(n,kount)
c
c     Function
c     --------
c
c        New work variable of length n, adapt counter kount
c
      integer n,kount
c
      kount=kount-n
      if (kount .le. 0) then
        write(*,*) 'Use larger work space arrays!'
        stop
      endif
      Nwork=kount+1
c
      return
      end



      integer function Kwork(n,kount)
c
c     Function
c     --------
c
c        Kill work variable of length n, adapt counter kount
c
      integer n,kount
c
      kount=kount+n
      Kwork=0
c
      return
      end



      integer function NumJ(n,info,Y,F,Jac,t,Nstep,Eval,error,
     +                      Dw,Iw,kountD,kountI)
c
      integer           n,info,error,kountD,kountI,Iw(*)
      double precision  Y(n),F(n),Jac(n,n),t,Nstep,Dw(*)
c
      external          Eval
c
c     Function
c     --------
c
c        Evaluate Jacobian by Numerical differencing (simple forward)
c
      integer           i,F2
      double precision  old,step,tiny,one
      parameter        (one=1.0d0,tiny=1d-30)
c
c     - Allocate another function value vector -
c
      F2=Nwork(n,kountD)
c
      do i=1,n
        old=Y(i)
        step=old*Nstep
        if (dabs(step) .le. tiny) step=Nstep
        Y(i)=old+step
        step=Y(i)-old
        call Eval(n,info,Dw(F2),Y,Jac,t,.false.,error,
     +            Dw,Iw,kountD,kountI)
        call SubVec(Dw(F2),Dw(F2),F,n)
        call ScVec(Dw(F2),one/step,Dw(F2),n)
        call PutMat(Jac,Dw(F2),i,n)
        Y(i)=old
      enddo
c
c     - Deallocate -
c
      F2=Kwork(n,kountD)
c
c     - Return results -
c
      NumJ=0
c
      return
      end



      subroutine AddBJ(Jac,B,infoB,n,nB)
c
c     Function
c     --------
c
c        Addition of matrix J and B
c
      integer n,nB,infoB(n)
      double precision Jac(n,n),B(nB,nB)
c
      integer i,j
c
      do i=1,n
        if (infoB(i) .ne. 0) then
          do j=1,n
            Jac(i,j)=Jac(i,j)+B(i,j)
          enddo
        endif
      enddo
c
      return
      end



      integer function Msirk3(n,k,info,infoB,Y,FF,B,nB,Jacc,h,t,Nstep,
     +                        Eval,Update,NumJac,UpdJac,iter,error,
     +                        Dw,Iw,kountD,kountI)
c
      integer           n,k,info,infoB(n),nB,iter,error,
     +                  kountD,kountI,Iw(*)
      double precision  Y(n),FF(n),B(nB,nB),Jacc(n,n),h,t,Nstep,Dw(*)
      logical           NumJac,UpdJac
c
      integer           Update
      external          Eval,Update
c
c     Function
c     --------
c
c        Perform one step with Generalized SIRK, with stepsize h,
c        results in Y.
c
      integer           singul
      double precision  a,b2,c2,b31,b32,R1,R2,R3,zero
      parameter        (singul=5,zero=0.0d0)
      parameter        (a  = 0.4358665215084590d0,
     +                  b2 = 0.75d0,
     +                  c2 = 0.84375d0,
     +                  b31=-0.6302020887244526d0,
     +                  b32=-0.2423378912600453d0,
     +                  R1 = 1.037609496131860d0,
     +                  R2 = 0.8349304838526379d0,
     +                  R3 = 1.0d0)
c
      integer           i,j,inx,Idid,F,Jac,k1,k2,k3,kc
      double precision  d
c
      kc=k
      Idid=0
c
c     - Allocate work variables -
c
      inx=Nwork(n,kountI)
      k1 =Nwork(n,kountD)
      k2 =Nwork(n,kountD)
      k3 =Nwork(n,kountD)
      F  =Nwork(n,kountD)
      Jac=Nwork(n*n,KountD)
c
c     - Copy Jacobian and function value vector -
c
      call CpyVec(Dw(F),FF,n)
      call CpyMat(Dw(Jac),Jacc,n)
c
c       - Evaluate F (and Jac) at Y -
c
 10     if (kc .lt. k) then
          call Eval(n,info,Dw(F),Y,Dw(Jac),t,
     +              (UpdJac .and. .not. NumJac),error,
     +              Dw,Iw,kountD,kountI)
          if ((error .eq. 0) .and. NumJac .and. UpdJac) then
            Idid=NumJ(n,info,Y,Dw(F),Dw(Jac),t,Nstep,Eval,error,
     +                Dw,Iw,kountD,kountI)
          endif
        endif
        if ((error .eq. 0) .and. (Idid .eq. 0)) then
c
c         - Inverse([B]-ha[J]): -
c
          if ((k .eq. kc) .or. UpdJac) then
            call ScaMat(Dw(Jac),-h*a,Dw(Jac),n)
            call AddBJ(Dw(Jac),B,infoB,n,nB)
            call LUdcmp(Dw(Jac),Iw(inx),d,n,Dw,kountD)
          else
            d=1.0d0
          endif
          if (d .eq. zero) then
            Idid=singul
          else
c
c           - k1: -
c
            call ScVec(Dw(k1),h,Dw(F),n)
            call LUbksb(Dw(Jac),Iw(inx),Dw(k1),n)
c
c           - k2 (evaluating F at Y+b2.k1): -
c
            call ScVec(Dw(F),b2,Dw(k1),n)
            call AddVec(Dw(k2),Y,Dw(F),n)
            call Eval(n,info,Dw(F),Dw(k2),Dw(Jac),t+c2*h,.false.,error,
     +                Dw,Iw,kountD,kountI)
            call ScVec(Dw(k2),h,Dw(F),n)
            call LUbksb(Dw(Jac),Iw(inx),Dw(k2),n)
c
c           - k3: -
c
            call ScVec(Dw(F),b31,Dw(k1),n)
            call ScVec(Dw(k3),b32,Dw(k2),n)
            call AddVec(Dw(F),Dw(F),Dw(k3),n)
            do i=1,n
              d=zero
              if (infoB(i) .ne. 0) then
                do j=1,n
                  d=d+B(i,j)*Dw(F+j-1)
                enddo
              endif
              Dw(k3+i-1)=d
            enddo
            call LUbksb(Dw(Jac),Iw(inx),Dw(k3),n)
c
c           - Y-next: -
c
            call ScVec(Dw(F),R1,Dw(k1),n)
            call ScVec(Dw(k2),R2,Dw(k2),n)
            call AddVec(Dw(F),Dw(F),Dw(k2),n)
            call ScVec(Dw(k3),R3,Dw(k3),n)
            call AddVec(Dw(F),Dw(F),Dw(k3),n)
c
c           - Update Y with F, and t with h ! -
c
            if (error .eq. 0) then
              Idid=Update(n,iter,info,t,h,Y,Dw(F))
            endif
          endif
        endif
c
c       - Decrement k counter -
c
        kc=kc-1
c
c     - Do next step? -
c
      if ((error .eq. 0) .and. (Idid .eq. 0) .and. (kc .gt. 0)) goto 10
c
c     - Deallocate work variables -
c
      Jac=Kwork(n*n,kountD)
      F  =Kwork(n,kountD)
      k3 =Kwork(n,kountD)
      k2 =Kwork(n,kountD)
      k1 =Kwork(n,kountD)
      inx=Kwork(n,kountI)
c
c     - Return result -
c
      Msirk3=Idid
c
      return
      end



      Subroutine Chk(Idid,test)
c
      integer Idid,test
c
c     Function
c     --------
c
c        Error check routine. Passes nonzero test into Idid.
c
      if (Idid .eq. 0) Idid=test
c
      return
      end



      integer function ChkTol(AbsTol,RelTol,n)
c
      integer           n
      double precision  AbsTol(n),RelTol(n)
c
c     Function
c     --------
c
c        Checks whether the tolerances are larger than minimum
c
      integer           tolerr,i,c
      double precision  mintol
      parameter        (tolerr=2,mintol=1.0d-14)
c
      c=0
      do i=1,n
        AbsTol(i)=dabs(AbsTol(i))
        RelTol(i)=dabs(RelTol(i))
        if (AbsTol(i)+RelTol(i) .lt. mintol) c=tolerr
      enddo
      ChkTol=c
c
      return
      end



      integer function ChkN(Nstep)
c
      double precision  Nstep
c
c     Function
c     --------
c
c        Check whether the step for the Numerical differencing is OK
c
      integer           nerr
      double precision  minN
      parameter        (nerr=2,minN=1.0d-10)
c
      if (Nstep .lt. minN) then
        ChkN=nerr
      else
        ChkN=0
      endif
c
      return
      end



      integer function GetB(n,info,B,nB,infoB)
c
      integer           n,info,nB,infoB(n)
      double precision  B(nB,nB)
c
c     Function
c     --------
c
c        Check the matrix B for nonzero rows. If row i is zero then
c        set infoB(i) to 0 else to 1. Skip unnecessary work. Check
c        whether B is totally empty; then inode is set!
c
      parameter        (node=10)
      integer           i,j,k,inode
c
      inode=node
      do i=1,n
        k=0
        j=1
 1      if (B(i,j) .ne. 0) then
          k=1
          inode=0
        endif
        if ((j .lt. n) .and. (k .eq. 0)) then
          j=j+1
          goto 1
        endif
        infoB(i)=k
        if (info .gt. 1) write(*,*) 'InfoB(',i,')=',infoB(i)
      enddo
      GetB=inode
c
      return
      end



      integer function RatPol(n,smax,suse,s,DidX,X,DidY,Y,dY,
     +                        Dw,Iw,kountD,kountI)
c
      integer           n,smax,suse,s,kountD,kountI,Iw(*)
      double precision  DidX(smax),X,DidY(n,smax),Y(n),dY(n),Dw(*)
c
c     Function
c     --------
c
c        Rational extraPolation for n functions:
c        Input: X and Y(n) estimates, history DidX and DidY
c        Output: Y(n) extrapolated to X->0 and error estimates dY(n)
c
      integer           j,k,m1,polerr,fx
      double precision  b,b1,c,ddy,v,yy,zero
      parameter        (zero=0.0d0,polerr=9)
c
c     - Allocate fx -
c
      fx=Nwork(smax,kountD)
c
c     - If first value in table -
c
      DidX(s)=X
      if (s .eq. 1) then
        do j=1,n
          DidY(j,1)=Y(j)
          dY(j)=Y(j)
        enddo
      else
c
c       - Extrapolation -
c
        if (s .lt. suse) then
          m1=s
        else
          m1=suse
        endif
        do k=1,m1-1
          Dw(fx+k+1-1)=DidX(s-k)/X
        enddo
        do j=1,n
          yy=Y(j)
          v=DidY(j,1)
          c=yy
          DidY(j,1)=yy
          do k=2,m1
            b1=Dw(fx+k-1)*v
            b=b1-c
            if (b .ne. zero) then
              b=(c-v)/b
              ddy=c*b
              c=b1*b
            else
              ddy=v
            endif
            if (k .ne. s) v=DidY(j,k)
            DidY(j,k)=ddy
            yy=yy+ddy
          enddo
          dY(j)=ddy
          Y(j)=yy
        enddo
      endif
c
c     - Deallocate
c
      fx=Kwork(smax,kountD)
c
      RatPol=0
c
      return
      end



      integer function PolPol(n,smax,suse,s,DidX,X,DidY,Y,dY,
     +                        Dw,Iw,kountD,kountI)
c
      integer           n,smax,suse,s,kountD,kountI,Iw(*)
      double precision  DidX(smax),X,DidY(n,smax),Y(n),dY(n),Dw(*)
c
c     Function
c     --------
c
c        Polynomial extraPolation for n functions:
c        Input: X and Y(n) estimates, history DidX and DidY
c        Output: Y(n) extrapolated to X->0 and error estimates dY(n)
c
      integer           j,k1,m1,d,polerr
      double precision  delta,f1,f2,q,one,large
      parameter        (one=1.0d0,large=1.0d20,polerr=9)
c
c     - Allocate d -
c
      d=Nwork(n,kountD)
c
      DidX(s)=X
      do j=1,n
        dY(j)=Y(j)
      enddo
c
c     - If first value in table -
c
      if (s .eq. 1) then
        do j=1,n
          DidY(j,1)=Y(j)
        enddo
      else
c
c       - Extrapolation -
c
        if (s .lt. suse) then
          m1=s
        else
          m1=suse
        endif
        do j=1,n
          Dw(d+j-1)=Y(j)
        enddo
        do k1=1,m1-1
          delta=one/(DidX(s-k1)-X)
          f1=X*delta
          f2=DidX(s-k1)*Delta
          do j=1,n
            q=DidY(j,k1)
            DidY(j,k1)=dY(j)
            delta=Dw(d+j-1)-q
            if (dabs(delta) .gt. large) then
              PolPol=polerr
              d=Kwork(n,kountD)
              return
            endif
            dY(j)=f1*delta
            Dw(d+j-1)=f2*delta
            Y(j)=Y(j)+dY(j)
          enddo
        enddo
        do j=1,n
          DidY(j,s)=dY(j)
        enddo
      endif
c
c     - Deallocate -
c
      d=Kwork(n,kountD)
c
      PolPol=0
c
      return
      end



      subroutine Restor(t,h,Y,Torg,Horg,Yorg,n)
c
      integer          n
      double precision t,h,Y(n),Torg,Horg,Yorg(n)
c
c     Function
c     --------
c
c        Restores original t, h and Y
c
      t=Torg
      h=Horg
      call CpyVec(Y,Yorg,n)
c
      return
      end



      subroutine BESIRK(n,info,Y,B,nB,t,tout,treprt,
     +                  h,hnext,hmin,hmax,Idid,
     +                  Nstep,Eval,Update,Toler,Report,
     +                  NumJac,MaxIt,Iter,AbsTol,RelTol,AlgEqn,
     +                  Dw,Iw,kountD,kountI)
c
      integer           n,info,nB,MaxIt,Iter,kountD,kountI,Iw(*),Idid
      double precision  Y(n),B(nB,nB),t,tout,treprt,h,hnext,hmin,
     +                  hmax,Nstep,AbsTol(n),RelTol(n),Dw(*)
      logical           NumJac,AlgEqn
c
      integer           Update,Report,ChkTol,ChkN,GetB,PolPol,RatPol
      logical           Toler
      external          Eval,Update,Toler,Report
c
c     Function
c     --------
c
c        DAE Solver, full matrix version.
c
      integer           F,AF,Yorg,Eps,Jac,DidY,DidX,sOld,sUsed,error,
     +                  i,s,sToUse,Echeck,infoB,de,ae,index
      double precision  Direc,Ecur,Eold,Horg,Torg,X,zero,one,two,defh
      logical           Convrg,Uviol,Reduct
      parameter        (zero=0.0d0,one=1.0d0,two=2.0d0,defh=1.0d-3)
      double precision  hn,tr,stime
      parameter        (stime=1.0d-10)
c
c     Errors
c
      integer           nreach,fail,singul,userv,
     +                  nexth,done,polerr,node,algv
      parameter        (nreach=1,fail=4,singul=5,userv=6,
     +                  nexth=7,done=8,polerr=9,node=10,algv=11)
c
c     Sequence/Extrapolation declarations
c
      integer           Emax,sMax,sUse,sStep,sSeq(12),
     +                  polord,poltyp,cIndex
      double precision  Shrink,Grow,MaxInc
      logical           UpdJac
      common   /Bsirk1/ Emax,sMax,sUse,sStep,sSeq,
     +                  polord,poltyp,cIndex,
     +                  Shrink,Grow,MaxInc,
     +                  UpdJac
c
c     - Initialize common parameters -
c
      if ((sMax .le. 0) .or. (sMax .gt. 12)) call Init3
c
c     - Allocate workspace -
c
      infoB=Nwork(n,kountI)
      F    =Nwork(n,kountD)
      Jac  =Nwork(n*n,kountD)
      Eps  =Nwork(n,kountD)
      Yorg =Nwork(n,kountD)
      DidY =Nwork(sMax*n,kountD)
      DidX =Nwork(sMax,kountD)
c
c     - Initialize -
c
      error =0
      Idid  =0
c!      Iter  =0
      Uviol =.false.
      Reduct=.false.
c
c     - Set sOld to our estimate, sStep to 12 -
c
      sStep=1
      sOld =sUse
c
c     - Checks -
c
      hmax=dabs(hmax)
      hmin=dabs(hmin)
      call Chk(Idid,ChkTol(AbsTol,RelTol,n))
      if (NumJac) call Chk(Idid,ChkN(Nstep))
      if ((tout-t) .gt. zero) Then
        Direc=one
      else
        Direc=-one
      endif
      treprt=dabs(treprt)
      if (treprt .eq. zero) then
        tr=tout
      else
        tr=t+Direc*treprt
      endif
      hn=zero
c
c     - If zero stepsize h, try to find a good h (as ODE system) -
c
      if (h .eq. zero) then
c
c       - Store Y and compute f(y+f(y))-f(y), restore Y -
c
        call CpyVec(Dw(Yorg),Y,n)
        call Eval(n,info,Dw(F),Y,Dw(Jac),t,.False.,error,
     +            Dw,Iw,kountD,kountI)
        call ScVec(Dw(F),Direc,Dw(F),n)
        call AddVec(Y,Y,Dw(F),n)
        call Eval(n,info,Dw(Eps),Y,Dw(Jac),t+Direc*defh,.False.,error,
     +            Dw,Iw,kountD,kountI)
        call SubVec(Dw(F),Dw(Eps),Dw(F),n)
        call CpyVec(Y,Dw(Yorg),n)
c
c       - select largest of f(y+f(y))-f(y) / e(y)  -
c
        Horg=zero
        do i=1,n
          h=AbsTol(i)+RelTol(i)*Y(i)
          if (h .ne. zero) then
            h=dabs(Dw(Eps+i-1)/h)
          endif
          if (h .gt. Horg) Horg=h
        enddo
c
c       - if unsuccessful use default stepsize -
c
        if (Horg .eq. zero) then
          h=Direc*defh
          if (info .gt. 0) write(*,*) 'Default stepsize used'
        else
c
c       - otherwise compute stepsize (euler=1->sirk=3,1->sSeq(sUse)) -
c
          h=(sSeq(sUse)*two/Horg)**(one/3.0d0)
          if (info .gt. 0) write(*,*) 'Stepsize determined as',h
        endif
      endif
c
c     - Get B matrix and check id no ode's -
c
      call Chk(Idid,GetB(n,info,B,nB,Iw(infoB)))
c
c     - All OK: lets go! -
c
      if (Idid .eq. 0) then
        call Vec0(Dw(Eps),n)
        call Chk(Idid,Report(n,Iter,info,t,treprt,Y,Dw(Eps)))
c
c         - Check for negative/zero treprt -
c
          treprt=dabs(treprt)
          if (treprt .eq. zero) tr=tout
c
c         - Increment iteration counter -
c
 10       iter=iter+1
          if (info .gt. 0) write(*,11) iter
 11       format(/,'Iteration=',i5)
c
c         - Reset error checks -
c
          Eold=0
          Echeck=0
c
c         - Set sequence counters -
c
          sUsed=0
          s=0
c
c         - Check if stepsize not too large -
c
          if ((hmax .ne. zero) .and. (dabs(h) .gt. hmax)) then
            if (info .gt. 0) write(*,*) 'Limited stepsize ',h,
     +                                  ' to maximum'
            h=Direc*hmax
          endif
c
c         - Check stepsize smaller than step to tout; adapt it -
c
          if ((dabs(tout) .gt. dabs(t)) .and.
     +        (dabs(h) .gt. dabs(tout-t))) then
            if (info .gt. 0) write(*,*) 'Stepsize set to obtain tout'
            h=Direc*dabs(tout-t)
          endif
c
c         - Check stepsize smaller than step to new treprt; adapt it -
c
          if (dabs(t+h) .gt. dabs(tr)) then
            if (info .gt. 0) write(*,*) 'Stepsize set to obtain report'
            if (hn .gt. treprt) then
              hn=treprt
            else
              hn=h
            endif
            h=Direc*dabs(tr-t)
          else
            hn=zero
          endif
c
c         - Check if stepsize not too small -
c
          if ((hmin .ne. zero) .and. (dabs(h) .lt. hmin)) Then
            if (Info .gt. 0) Write(*,*) 'Increased stepsize ',h,
     +                                  ' to minimum'
            h=Direc*hmin
          endif
c
c         - Save original t and Y -
c
          Torg=t
          Horg=h
          call CpyVec(Dw(Yorg),Y,n)
c
c         - Evaluate first function values and Jacobian -
c
          call Eval(n,info,Dw(F),Y,Dw(Jac),t,(.not. NumJac),error,
     +              Dw,Iw,kountD,kountI)
          if ((error .eq. 0) .and. NumJac) then
            call Chk(Idid,
     +               NumJ(n,info,Y,Dw(F),Dw(Jac),t,Nstep,Eval,error,
     +               Dw,Iw,kountD,kountI))
          endif
c
c         - first iteration: check index -
c
          index=0
          if (cIndex .gt. 0) then
            if ((cIndex .gt. 1) .or. (iter .eq. 1)) then
              call IIndex(n,info,Iw(infoB),AbsTol,RelTol,Y,Dw(F),
     +                    Dw(Jac),de,ae,index,Iw,kountI)
              if (info .gt. 0) then
                write(*,*) 'Differential eqns.  ',de
                write(*,*) 'Algebraic eqns.     ',ae
                write(*,*) 'Indication of index ',index
              endif
            endif
          endif
c
          if (index .lt. 0) then
            Idid=algv
            write(*,*) '* Warning; Algebraic eqn(s) not satisfied *'
            index=-index
          endif
c
          if (index .gt. 1) then
            write(*,*) 'WARNING, index =',index,' > 1!'
          endif
c
c         - Integrate -
c
 20         if (sUsed .gt. 0) then
              call Restor(t,h,Y,Torg,Horg,Dw(Yorg),n)
            endif
c
c           - Increment sequence counter -
c
            sUsed=sUsed+1
            if (Info .gt. 0) Write(*,*) 'Using ',sSeq(sUsed),' steps'
c
c           - Set appropiate h -
c
            h=Horg/(sSeq(sUsed))
c
c           - Do Integration(h)
c
            call Chk(Idid,Msirk3(n,sSeq(sUsed),info,Iw(infoB),Y,Dw(F),
     +                           B,nb,Dw(Jac),h,t,NStep,Eval,Update,
     +                           NumJac,UpdJac,iter,error,
     +                           Dw,Iw,kountD,kountI))
c
c           - Check for User violation -
c
            if (error .eq. userv) then
              Uviol=.true.
c
c             - Reset tout! -
c
              tout=t
              horg=tout-Torg
              if (Horg .eq. zero) Horg=sSeq(sUsed)*h/2.0d0
c
c             - Check if we can try to reach upto user violation -
c
              if (dabs(Horg) .gt. hmin) then
c
c               - Reset this step and redo -
c
                error=0
                tout=Torg+Horg
                s=0
                sUsed=0
                Convrg=.false.
                call Restor(t,h,Y,Torg,Horg,Dw(Yorg),n)
                if (info .gt. 0) then
                  write(*,*) 'User function violation,',
     +                       ' temporary tout=',tout
                endif
              endif
            else
c
c             - Check for Singularity -
c
              if (Idid .eq. singul) then
                if (info .gt. 0) write(*,*) 'Singularity: next step'
c
c               - force us to use next in sequence -
c
                Convrg=.false.
                sUsed=sUsed+1
                Idid=0
              else
c
c               - Increment steps in extrapolation -
c
                s=s+1
c
c               - Right order in stepsize -
c
                X=h
                if (polord .eq. 0) polord=4
                do i=2,polord
                  X=X*h
                enddo
c
c               - Calculate error, and approximation Y -
c
                if (poltyp .eq. 0) then
                  call Chk(Idid,PolPol(n,sMax,sUse,s,Dw(DidX),X,
     +                                 Dw(DidY),Y,Dw(Eps),
     +                                 Dw,Iw,kountD,kountI))
                else
                  call Chk(Idid,RatPol(n,sMax,sUse,s,Dw(DidX),X,
     +                                 Dw(DidY),Y,Dw(Eps),
     +                                 Dw,Iw,kountD,kountI))
                endif
c
c               - Check polation error
c
                if (Idid .eq. polerr) then
                  if (info .gt. 0) write(*,*) 'Polation error'
                  Idid=0
                  Echeck=Emax
                endif
c
c               - See if we want to control algebraic equations -
c
                if (AlgEqn) then
                  AF=Nwork(n,kountD)
                  call Eval(n,info,Dw(AF),Y,Dw(Jac),t,.False.,error,
     +                      Dw,Iw,kountD,kountI)
                  do i=1,n
                    if (Iw(infoB+i-1) .eq. 0) then
                      Dw(Eps+i-1)=Dw(AF+i-1)
                    endif
                  enddo
                  AF=Kwork(n,kountD)
                endif
c
c               - Check error with tolerance, if we have done>1 steps -
c
                if ((sUsed .gt. 1) .and. (Echeck .lt. Emax)) then
                  Convrg=Toler(n,iter,info,t,Ecur,Y,Dw(Eps),
     +                         AbsTol,RelTol)
                  if ((.not. Convrg) .and.
     +                (sOld .ne. 0) .and.
     +                (sUsed .gt. sOld+sStep)) then
                    Echeck=Emax
                  else
                    if (Eold .eq. 0) Then
                      Eold=Ecur
                    else
                      if (Eold .gt. Ecur) then
                        Echeck=0
                        Eold=Ecur
                      else
                        Echeck=Echeck+1
                      endif
                    endif
                  endif
                else
                  Convrg=.false.
                endif
              endif
            endif
c
c           - stop when converged, max steps, or error -
c
            if ((Convrg) .or.
     +          (sUsed .eq. sMax) .or.
     +          (error .ne. 0) .or.
     +          (Idid .ne. 0) .or.
     +          (Echeck .ge. Emax)) goto 30
c
            goto 20
c
c         - Make sure no round-off in stepping in t occurs -
c
 30       if (Convrg) then
            t=Torg+Horg
          endif
c
c         - Signal failed step -
c
          if ((Echeck .ge. Emax) .or.
     +        (.not. Convrg .and. (sUsed .eq. sMax))) then
            if (Info .gt. 0) write(*,*) 'Failed Step'
            Idid=fail
          endif
c
c         - Signal errors -
c
          if (((Idid .ne. 0) .or. (error .ne. 0)) .and.
     +        (error .ne. userv)) then
c
c           - Restore original t and Y -
c
            call Restor(t,h,Y,Torg,Horg,Dw(Yorg),n)
            if (Info .gt. 0) then
              write(*,*) 'Error=',error,' Idid=',Idid,
     +                   ' Original t and Y restored'
            endif
          endif
c
c         - Compute next step, hnext. If successfull integration: -
c
          if (Idid .eq. 0) then
c
c           - Get the number of steps to have been used for this step -
c
            sToUse=Min(sOld+sStep,sUse)
c
c           - Calculate new stepsize -
c
            if (sUsed .eq. sToUse) then
              if (info .gt. 0) Write(*,*) 'Stepsize a little smaller'
              hnext=Shrink*horg
            else
              if (sUsed .eq. sToUse-1) then
                if (info .gt. 0) Write(*,*) 'Stepsize a little larger'
                hnext=grow*horg
              else
                if (info .gt. 0) then
                  if (sToUse .gt. sUsed) then
                    write(*,*) 'Increased stepsize'
                  else
                    write(*,*) 'Decreased stepsize'
                  endif
                endif
                hnext=Horg*dble(sSeq(sToUse-1))/dble(sSeq(sUsed))
                if (hnext/Horg .gt. MaxInc) hnext=Horg*MaxInc
              endif
            endif
          else
c
c           - If stepfailure -
c
            if (Idid .eq. fail) then
              if (Info .gt. 0) write(*,*) 'Stepsize reduction'
              hnext=Horg/4.0d0
              do i=1,(sMax-sUse)/2
                hnext=hnext/2.0d0
              enddo
            else
              hnext=h
            endif
          endif
c
c         - Step Reduction? Prevent step increment after a reduction.
c
          if (hnext .lt. horg) then
            Reduct=.true.
          else
            If (Reduct) then
              if (Info .gt. 0) write(*,*) 'Restricted step increment'
              hnext=horg
            endif
            Reduct=.false.
          endif
c
c         - Remember number of steps used -
c
          if (sOld .eq. sUsed) then
            sStep=2
          else
            sStep=1
          endif
          sOld=sUsed
c
c         - Check next stepsize
c
          if (dabs(hnext) .lt. hmin) then
            Idid=nexth
            if (Info .gt. 0) then
              write(*,*) 'Error: next stepsize<minimum',
     +                   ', can not continue'
            endif
          endif
c
c         - Check whether we had a user violation -
c
          if (Uviol) then
            Idid=userv
          else
c
c           - Check if last step was to obtain a report time,
c             if so use the previously computed hnext (hn) -
c
            if ((hn .ne. zero) .and. (hnext .gt. horg)) then
              hnext=hn
            endif
            if (Info .gt. 0) write(*,*) 'Next stepsize=',hnext
            h=hnext
c
c           - Report at report time AND at tout -
c
            if ((dabs(t) .ge. dabs(tout-stime)) .or.
     +          (dabs(t-tr) .lt. stime)) then
              call Chk(Idid,Report(n,iter,info,t,treprt,Y,Dw(Eps)))
c
c             - Check for zero treprt -
c
              treprt=dabs(treprt)
              if (treprt .eq. zero) then
                tr=tout
              else
c
c               - New report time -
c
                if (dabs(t-tr) .lt. stime) then
                  tr=tr+Direc*treprt
                endif
              endif
c
c             - Check if we're done else if OK use next stepsize -
c
              if (dabs(t) .ge. dabs(tout-stime)) then
                call Chk(Idid,done)
              endif
            endif
          endif
c
c         --- If failed try again ---
c
          if (Idid .eq. fail) Idid=0
c
c         - Stop on error or when maximum number of iterations
c
          if ((iter .eq. MaxIt) .or.
     +        (error .ne. 0) .or.
     +        (Idid .ne. 0)) goto 40
c
          goto 10
c
c       - Check if we reached tout -
c
 40     if (t .ne. tout) call Chk(Idid,nreach)
c
c       - Check error/Idid -
c
        if (error .ne. 0) then
          Idid=-abs(error)
        endif
c
c       - Check if we were done -
c
        if (Idid .eq. done) Idid=0
c
      endif
c
c     - Deallocate workspace -
c
      DidX =Kwork(sMax,kountD)
      DidY =Kwork(sMax*n,kountD)
      Yorg =Kwork(n,kountD)
      Eps  =Kwork(n,kountD)
      Jac  =Kwork(n*n,kountD)
      F    =Kwork(n,kountD)
      infoB=Kwork(n,kountI)
c
      return
      end



      subroutine Init1
c
c     Function
c     --------
c
c        Sequence/Extrapolation declarations for BESIRK order 1.
c
      integer           Emax,sMax,sUse,sStep,sSeq(12),
     +                  polord,poltyp,cIndex
      double precision  Shrink,Grow,MaxInc
      logical           UpdJac
      common   /Bsirk1/ Emax,sMax,sUse,sStep,sSeq,
     +                  polord,poltyp,cIndex,
     +                  Shrink,Grow,MaxInc,
     +                  UpdJac
c
c     - Initialize to default primes sequence -
c
      Emax=2
      sMax=12
      sUse=10
c
      Shrink=0.9d0
      Grow=1.1d0
      MaxInc=10.0d0
c
      sSeq(1)=1
      sSeq(2)=2
      sSeq(3)=3
      sSeq(4)=5
      sSeq(5)=7
      sSeq(6)=11
      sSeq(7)=17
      sSeq(8)=23
      sSeq(9)=31
      sSeq(10)=47
      sSeq(11)=61
      sSeq(12)=89
c
c     - Set default polynomial extrapolation -
c
      polord=1
      poltyp=0
      cIndex=1
      UpdJac=.False.
c
      return
      end



      subroutine Init3
c
c     Function
c     --------
c
c        Sequence/Extrapolation declarations for BESIRK order 3.
c
      integer           Emax,sMax,sUse,sStep,sSeq(12),
     +                  polord,poltyp,cIndex
      double precision  Shrink,Grow,MaxInc
      logical           UpdJac
      common   /Bsirk1/ Emax,sMax,sUse,sStep,sSeq,
     +                  polord,poltyp,cIndex,
     +                  Shrink,Grow,MaxInc,
     +                  UpdJac
c
c     - Initialize to default primes sequence -
c
      Emax=2
      sMax=12
      sUse=11
c
      Shrink=0.9d0
      Grow=1.1d0
      MaxInc=10.0d0
c
      sSeq(1)=1
      sSeq(2)=2
      sSeq(3)=3
      sSeq(4)=5
      sSeq(5)=7
      sSeq(6)=11
      sSeq(7)=17
      sSeq(8)=23
      sSeq(9)=31
      sSeq(10)=47
      sSeq(11)=61
      sSeq(12)=89
c
c     - Set default polynomial extrapolation -
c
      polord=3
      poltyp=0
      cIndex=1
      UpdJac=.True.
c
      return
      end



      subroutine Newton(n,info,Y,t,Idid,Nstep,Eval,Update,Toler,
     +                  Report,NumJac,UpdJac,MaxIt,Iter,
     +                  AbsTol,RelTol,Fnorm,
     +                  Dw,Iw,kountD,kountI)
c
      integer           n,info,Idid,MaxIt,Iter,kountD,kountI,Iw(*)
      double precision  Y(n),t,Nstep,AbsTol(n),RelTol(n),Fnorm,Dw(*)
      logical           NumJac,UpdJac
c
      integer           Update,Report,ChkTol,ChkN
      logical           Toler
      external          Eval,Update,Toler,Report
c
c     Function
c     --------
c
c        Solves the "time invariant" problem with the Newton's method
c
      integer           F,Jac,inx,i,error,singul
      double precision  d,h
      logical           needJ,done
      parameter        (singul=5)
c
c     - Allocate workspace -
c
      inx=Nwork(n,kountI)
      F  =Nwork(n,kountD)
      Jac=Nwork(n*n,kountD)
c
c     - Initialize -
c
      h=0
c!      Iter=0
      Idid=0
      error=0
c
c     - Checks -
c
      Idid=ChkTol(AbsTol,RelTol,n)
      if (NumJac) Idid=ChkN(Nstep)
c
 10     Idid=Report(n,Iter,Info,t,t,Y,Dw(F))
        needJ=(UpdJac .or. (Iter .eq. 0))
c
c       - Evaluate F (and Jacobian) at Y,t -
c
        call Eval(n,info,Dw(F),Y,Dw(Jac),t,
     +            (needj .and. .not. NumJac),error,
     +            Dw,Iw,kountD,kountI)
        Fnorm=0.0d0
        do i=0,n-1
          Fnorm=Fnorm+Dw(F+i)**2
        enddo
        Fnorm=dsqrt(Fnorm)
        if ((error .eq. 0) .and. (Idid .eq. 0)) then
          done=Toler(n,Iter,info,t,d,Y,Dw(F),AbsTol,RelTol)
          if (.not. done) then
            if (needJ .and. NumJac .and. (error .eq. 0)) then
              Idid=NumJ(n,info,Y,Dw(F),Dw(Jac),t,Nstep,Eval,error,
     +                  Dw,Iw,kountD,kountI)
            endif
            if (needJ) then
              call LUdcmp(Dw(Jac),Iw(inx),d,n,Dw,kountD)
            else
              d=1
            endif
            if (d .eq. 0) then
              Idid=singul
            else
              call LUbksb(Dw(Jac),Iw(inx),Dw(F),n)
              call ScVec(Dw(F),-1.0d0,Dw(F),n)
              Idid=Update(n,Iter,info,t,h,Y,Dw(F))
              Iter=Iter+1
            endif
          endif
        endif
        if ((.not. done) .and.
     +      (Idid .eq. 0) .and.
     +      (error .eq. 0) .and.
     +      (Iter .le. MaxIt)) goto 10
c
c     - Deallocate -
c
      Jac=Kwork(n*n,kountD)
      F  =Kwork(n,kountD)
      inx=Kwork(n,kountI)
c
      if (Iter .gt. MaxIt) then
        call Chk(Idid,1)
      endif
c
      return
      end



c
c     ================== Standard Update/Toler/Report =================
c

      integer function StandU(n,iter,info,t,h,Y,DeltaY)
c
      integer           n,iter,info
      double precision  t,h,Y(n),DeltaY(n)
c
c     Function
c     --------
c
c        Standard update routine for BESIRK.
c
      t=t+h
      call AddVec(Y,Y,DeltaY,n)
      if (info .gt. 0) write(*,10)
      StandU=0
c
      return
   10 format('.',\)
      end



      logical function StandT(n,iter,info,t,et,Y,E,A,R)
c
      integer           n,iter,info
      double precision  t,et,Y(n),E(n),A(n),R(n)
c
c     Function
c     --------
c
c        Standard tolerance check routine for BESIRK.
c
      logical           done
      integer           i
      double precision  o,l,tiny
      parameter        (tiny=1.0d-30)
c
      done=.true.
      et=tiny
      do i=1,n
        o=dabs(E(i))
        l=A(i)+R(i)*dabs(Y(i))+tiny
        et=max(et,o/l)
        if (o .gt. l) then
          done=.false.
          if (info .gt. 0) write(*,10) i
        endif
      enddo
      StandT=done
      if (info .gt. 0) then
        if (done) then
          write(*,*) ' ',et,' t=',t
        else
          write(*,*) ' ',et
        endif
      endif
c
      return
   10 format(' ',i6,\)
      end



      integer function StandR(n,iter,info,t,treprt,Y,E)
c
      integer           n,iter,info
      double precision  t,Y(n),E(n),treprt
c
c     Function
c     --------
c
c        Standard report routine for BESIRK.
c
      integer           i
c
      if (info .gt. 0) then
        write(*,*) 'Report'
        write(*,*) 'time=',t
        do i=1,n
          write(*,*) 'Y',i,'=',Y(i)
        enddo
      endif
      StandR=0
c
      return
      end



c
c     ============================ BSIRK ===============================
c

      subroutine BSIRK(n,info,Y,B,nB,t,tout,treprt,
     +                 h,hnext,hmin,hmax,Idid,
     +                 Nstep,Eval,
     +                 NumJac,MaxIt,Iter,AbsTol,RelTol,
     +                 Dw,Iw,kountD,kountI)
c
      integer           n,info,nB,MaxIt,Iter,kountD,kountI,Iw(*),Idid
      double precision  Y(n),B(nB,nB),t,tout,treprt,h,hnext,hmin,
     +                  hmax,Nstep,AbsTol(n),RelTol(n),Dw(*)
      logical           NumJac
c
c     Function
c     --------
c
c        BESIRK with standard routines.
c
      integer           StandU,StandR
      logical           StandT
      external          Eval,StandU,StandT,StandR
c
      call BESIRK(n,info,Y,B,nB,t,tout,treprt,
     +            h,hnext,hmin,hmax,Idid,
     +            Nstep,Eval,StandU,StandT,StandR,
     +            NumJac,MaxIt,Iter,AbsTol,RelTol,.False.,
     +            Dw,Iw,kountD,kountI)
c
      return
      end



c
c     ============================= IINDEX ============================
c

      subroutine IIndex(n,info,infoB,AbsTol,RelTol,Y,F,
     +                  Jac,de,ae,index,Iw,kountI)
c
      integer           n,info,infoB(n),de,ae,index,Iw(*)
      double precision  AbsTol(n),RelTol(n),Y(n),F(n),Jac(n,n)
c
c     Function
c     --------
c
c        DAE index check routine
c
      integer           i,j,k,isign,inx,help1,help2
      logical           UsedA
      double precision  tol
c
c     Count type of equations, check algebraic constraints
c
      de=0
      ae=0
      isign=1
      do i=1,n
        if (infoB(i) .eq. 0) then
          ae=ae+1
          tol=AbsTol(i)+RelTol(i)*dabs(Y(i))
          if (dabs(F(i)) .gt. tol) then
            isign=-1
            write(*,*) 'Unsatisfied algebraic eqn#',i
            write(*,*) 'Equation value = ',F(i)
          endif
        else
          de=de+1
        endif
      enddo
c
c     Check algebraic equations for index problems
c
      if (ae .eq. 0) then
        index=0
      else
        if (de .eq. 0) then
          index=-1
        else
          index=1
        endif
      endif
c
      if (index .eq. 1) then
        help1=Nwork(n,kountI)
        help2=Nwork(n,kountI)
        do i=1,n
          if (infoB(i) .eq. 0) then
            if (Jac(i,i) .eq. 0.0d0) then
c
c             Assume index = 1
c
              inx=1
c
c             Make status vector
c
              do j=1,n
                if (Jac(i,j) .eq. 0.0d0) then
                  Iw(help2+j-1)=0
                else
                  Iw(help2+j-1)=1
                endif
              enddo
c  loop
 1              UsedA=.false.
c
c               Use algebraic equations
c
                do j=1,n
                  if ((i .ne. j) .and. (infoB(j) .eq. 0)) then
                    if (Jac(i,j) .ne. 0.0d0) then
                      do k=1,n
                        if (Jac(j,k) .ne. 0.0d0) then
                          if (Iw(help2+k-1) .eq. 0) then
                            Iw(help2+k-1)=j
                            UsedA=.true.
                          endif
                        endif
                      enddo
                    endif
                  endif
                enddo
c
c               Repeat until all algebraic possibilities are used up
c
                if (UsedA) goto 1
c
c               Copy current status vector
c
                do j=1,n
                  Iw(help1+j-1)=Iw(help2+j-1)
                enddo
c
c               Check status vector if equation is reduced
c
                if ((Iw(help1+i-1) .eq. 0) .and. (inx .lt. n)) then
c
c                 Increment index
c
                  inx=inx+1
c
c                 Use differential equations
c
                  do j=1,n
                    if (infoB(j) .ne. 0) then
                      if (Iw(help1+j-1) .ne. 0) then
                        do k=1,n
                          if (Jac(j,k) .ne. 0.0d0) then
                            if (Iw(help2+k-1) .eq. 0) then
                              Iw(help2+k-1)=j
                            endif
                          endif
                        enddo
                      endif
                    endif
                  enddo
c
c                 Loop back
c
                  goto 1
                endif
c
c             Largest index found
c
              if (inx .gt. index) index=inx
c
c             Report substitutions:
c
              if (info .gt. 0) then
c
c               Reset original dependence in status vector
c
                do j=1,n
                  if (Jac(i,j) .ne. 0.0d0) then
                    Iw(help2+j-1)=0
                  endif
                enddo
cc
cc               Dump of status vector
cc
c                do j=1,n
c                  k=Iw(help2+j-1)
c                  if (k .ne. 0) then
c                    write(*,*) j,k
c                  endif
c                enddo
cc
c               Start with i-th eqn, backtrack trough the status
c               vector until an orginal dependence is found,
c               reporting both equation and variable:
c
                write(*,*) '-Index problem-'
                write(*,*) ' Index:     #'
                write(*,3) ' ',inx,i
                write(*,*) ' Substitutions:'
 2              j=Iw(help2+j-1)
                if (j .ne. 0) then
                  if (infoB(j) .eq. 0) then
                    write(*,3) '  ',inx,j
                  else
                    inx=inx-1
                    write(*,3) ' *',inx,j
                  endif
 3                format((A),i5,i6)
                  goto 2
                endif
                write(*,*) '---------------'
              endif
c
            endif
          endif
        enddo
        help2=Kwork(n,kountI)
        help1=Kwork(n,kountI)
      endif
c
      index=isign*index
c
      return
      end



c
c     ======================== Utility routines =======================
c

      subroutine CpyVec(Vout,Vin,n)
c
c     Function
c     --------
c
c        Copy vector Vin (size n) into Vout
c
      integer          n
      double precision Vout(n),Vin(n)
c
      integer          i
c
      do i=1,n
        Vout(i)=Vin(i)
      enddo
c
      return
      end



      subroutine CpyMat(Mout,Min,n)
c
c     Function
c     --------
c
c        Copy matrix Min (size nmax) into Mout
c
      integer          n
      double precision Mout(n,n),Min(n,n)
c
      integer          i,j
c
      do i=1,n
        do j=1,n
          Mout(i,j)=Min(i,j)
        enddo
      enddo
c
      return
      end



      subroutine Vec0(V,n)
c
c     Function
c     --------
c
c        Reset vector V (size n) to 0
c
      integer          n
      double precision V(n)
c
      integer          i
c
      do i=1,n
        V(i)=0.0d0
      enddo
c
      return
      end



      subroutine Mat0(M,nd,n)
c
c     Function
c     --------
c
c        Reset matrix (size n*n) to 0
c
      integer          n
      double precision M(nd,nd)
c
      integer          i,j
c
      do i=1,n
        do j=1,n
          M(i,j)=0.0d0
        enddo
      enddo
c
      return
      end



      subroutine AddVec(Vout,V1,V2,n)
c
c     Function
c     --------
c
c        Add vectors (size n) V1 and V2 into Vout
c
      integer          n
      double precision Vout(n),V1(n),V2(n)
c
      integer          i
c
      do i=1,n
        Vout(i)=V1(i)+V2(i)
      enddo
c
      return
      end



      subroutine SubVec(Vout,V1,V2,n)
c
c     Function
c     --------
c
c        Subtract vector (size n) V2 from V1, put result into Vout
c
      integer          n
      double precision Vout(n),V1(n),V2(n)
c
      integer          i
c
      do i=1,n
        Vout(i)=V1(i)-V2(i)
      enddo
c
      return
      end



      subroutine ScVec(Vout,Scalar,Vin,n)
c
c     Function
c     --------
c
c        Scale vector Vin (size n) and put result into Vout
c
      integer          n
      double precision Vout(n),Scalar,Vin(n)
c
      integer          i
c
      do i=1,n
        Vout(i)=Scalar*Vin(i)
      enddo
c
      return
      end



      subroutine ScaMat(Mout,Scalar,Min,n)
c
c     Function
c     --------
c
c        Scale matrix Min (size n by n) and put result into Mout
c
      integer          n
      double precision Mout(n,n),Scalar,Min(n,n)
c
      integer          i,j
c
      do i=1,n
        do j=1,n
          Mout(i,j)=Scalar*Min(i,j)
        enddo
      enddo
c
      return
      end



      subroutine PutMat(Mout,Vin,Column,n)
c
c     Function
c     --------
c
c        Put vector Vin (size n) in Column of matrix Mout (size n by n)
c
      integer          Column,n
      double precision Mout(n,n),Vin(n)
c
      integer          i
c
      do i=1,n
        Mout(i,Column)=Vin(i)
      enddo
c
      return
      end



c
c     =============== Routines from the Numerical Recipes =============
c

      subroutine LUdcmp(a,indx,d,n,Dw,kountD)
c
      integer           n,indx(n),kountD
      double precision  a(n,n),d,Dw(*)
c
c     Function
c     --------
c
c        LU matrix decomposition
c
      integer           i,imax,j,k,vv
      double precision  aamax,dum,sum,zero,one,tiny
      parameter        (zero=0.0d0,one=1.0d0,tiny=1.0d-20)
c
c     - Allocate workspace vv -
c
      vv=Nwork(n,kountD)
c
c     - Obtain row scaling, save in vv -
c
      d=one
      do i=1,n
        aamax=zero
        do j=1,n
          dum=dabs(a(i,j))
          if (dum .gt. aamax) aamax=dum
        enddo
        if (aamax .eq. zero) then
          d=zero
          vv=Kwork(n,kountD)
          return
        endif
        Dw(vv+i-1)=one/aamax
      enddo
c
      do j=1,n
        do i=1,j-1
          sum=a(i,j)
          do k=1,i-1
            sum=sum-a(i,k)*a(k,j)
          enddo
          a(i,j)=sum
        enddo
        aamax=zero
        do i=j,n
          sum=a(i,j)
          do k=1,j-1
            sum=sum-a(i,k)*a(k,j)
          enddo
          a(i,j)=sum
          dum=Dw(vv+i-1)*dabs(sum)
          if (dum .ge. aamax) then
            imax=i
            aamax=dum
          endif
        enddo
        if (j .ne. imax) then
          do k=1,n
            dum=a(imax,k)
            a(imax,k)=a(j,k)
            a(j,k)=dum
          enddo
          d=-d
          Dw(vv+imax-1)=Dw(vv+j-1)
        endif
        indx(j)=imax
        if (a(j,j) .eq. zero) a(j,j)=tiny
        if (j .ne. n) then
          dum=one/a(j,j)
          do i=j+1,n
            a(i,j)=a(i,j)*dum
          enddo
        endif
      enddo
c
c     - Normal return, deallocate vv -
c
      vv=Kwork(n,kountD)
c
      return
      end



      subroutine LUbksb(a,indx,b,n)
c
      integer           n,indx(n)
      double precision  a(n,n),b(n)
c
c     Function
c     --------
c
c        LU back-substitution
c
      integer           i,ii,j,ll
      double precision  sum,zero
      parameter        (zero=0.0d0)
c
      ii=0
      do i=1,n
        ll=indx(i)
        sum=b(ll)
        b(ll)=b(i)
        if (ii .ne. 0) then
          do j=ii,i-1
            sum=sum-a(i,j)*b(j)
          enddo
        else if (sum .ne. zero) then
          ii=i
        endif
        b(i)=sum
      enddo
      do i=n,1,-1
        sum=b(i)
        do j=i+1,n
          sum=sum-a(i,j)*b(j)
        enddo
        b(i)=sum/a(i,i)
      enddo
c
      return
      end
